/******************************************************************************
 * \file: trace_interface.h
 *
 * \version: $Id: trace_interface.h,v 1.15 2011/11/11 12:01:01 arun.v Exp $
 *
 * This file exports application interface by Trace component
 *
 * \component: Gen2 Trace
 *
 * \author Sakthivelu.S
 *
 * \copyright: (c) 2003 - 2009 ADIT
 *
 *****************************************************************************/
#ifndef _TRACE_INTERFACE_H
#define _TRACE_INTERFACE_H

#include <adit_typedef.h>
#include <adit_trace.h>

#ifdef __cplusplus
extern "C" {
#endif

#ifdef ETG_ENABLED
/**
 * Init Platform Trace to ETG
 *
 * \parm  void
 * return \li void
 */
IMPORT void vInitPlatformEtg(void);
#endif

#define TRACE_LARGE_MSG   241

/* Trace Callback */
typedef void (*TRACE_CALLBACK)(VP);

/* Class/level enable status */
typedef enum
{
  TR_ENABLED_CLASS_NONE   = 0x00,
  TR_ENABLED_CLASS_TRC    = 0x01,
  TR_ENABLED_CLASS_PXY    = 0x02,
  TR_ENABLED_CLASS_PMM    = 0x04,
  TR_ENABLED_CLASS_ALL    = 0x07
}TRACE_clas_ena_stat;

typedef enum
{
  TRACE_APP_FILTER  = 0x00,
  TRACE_TRIP_STATUS = 0x01,
  TRACE_MAX_NOTIFY_EVT
}TRACE_notify_evt;

/* Trace appfilter notification */
typedef struct
{
  TRACE_notify_evt  evt_id;               /* notification event */
  TRACE_CALLBACK    func;                 /* Callback fn to receive notifications */
}TRACE_notify;

typedef enum
{
  TRACE_UDA0 = 0x0,
  TRACE_UDA1 = 0x1
}TRACE_media_type;

/*Config structure for saving configuration on FFD or other early storage devices*/
/**************************************************/

/* Application interface API */

/**
 * trace out message when class/level is enabled
 *
 * \parm  uwLen       Length of the msg in bytes
 *        trClass     Trace class of the msg
 *        trLevel     Trace level of the msg
 *        p_ubTrData  ptr to a byte buffer containing msg
 * return \li U32      number of bytes traced out
 */
IMPORT U32 TR_core_uwTraceOut(U32 uwLen, U16 trClass, TR_tenTraceLevel trLevel,
                                                            U8* p_ubTrData);

/**
 * Trace out MST messages
 *
 * \parm  uwCompId        Component ID of the MST tester class
 *        uwSockId        socketID as received from MST client
 *        uwBufferLen     Length of the msg in bytes
 *        p_ubDataBuffer  ptr to a byte buffer containing msg
 * return \li U32          number of bytes traced out
 */
IMPORT U32 TR_core_uwTraceBinOutput(U32 uwCompId, U32 uwSockId, U32 uwBufferLen,
                                   U8* p_ubDataBuffer);
/**
 * checking class/level status
 *
 * \parm  trClass        Trace class
 *        enLevel        Trace level
 * return \li BOOL       TR_ENABLED_CLASS_TRC, if enabled for TTFis
 *                       TR_ENABLED_CLASS_PXY, if enabled for Proxy
 *                       TR_ENABLED_CLASS_PMM, if enabled for PMM
 *                       TR_ENABLED_CLASS_ALL, if enabled for all above
 *                       TR_ENABLED_CLASS_NONE, otherwise
 */
IMPORT BOOL TR_core_bIsClassSelected(U16 trClass, TR_tenTraceLevel enLevel);

/**
 * Registering channel callback with Trace
 *
 * \parm  chan_id        Trace Channel ID
 *        p_Func         Callback func
 * return \li BOOL       TRUE, if registered successfully
 *                       FALSE, otherwise
 */
IMPORT BOOL TR_chan_acess_bRegChan(TR_tenTraceChan chan_id,
                                   TRACE_CALLBACK p_Func);
/**
 * Unregister channel callback with Trace
 *
 * \parm  chan_id        Trace Channel ID
 *        p_Func         Callback func
 * return \li BOOL       TRUE, if unregistered successfully
 *                       FALSE, otherwise
 */
IMPORT BOOL TR_chan_acess_bUnRegChan(TR_tenTraceChan chan_id,
                                     TRACE_CALLBACK p_Func);
/**
 * This function is added for backward compatibility
 * trace out messages to Proxy
 *
 * \parm  uwLen       Length of the msg in bytes
 *        trClass     Trace class of the msg
 *        trLevel     Trace level of the msg
 *        p_ubTrData  ptr to a byte buffer containing msg
 * return \li U32      number of bytes traced out
 */
IMPORT U32 TR_proxy_uwTrace(U32 uwLen, TR_tenTraceClass trClass,
                           TR_tenTraceLevel trLevel, U8* p_ubTrData);
/**
 * This function is added for backward compatibility
 * check class/level status for Proxy
 *
 * \parm  enClass     Trace class
 *        enLevel     Trace level
 * return \li BOOL    TRUE, if enabled
 *                    FALSE, otherwise
 */
IMPORT BOOL TR_proxy_bIsClassSelected(TR_tenTraceClass enClass,
                                      TR_tenTraceLevel enLevel);
/*jov1kor-CRQ.No-XXXX- Implementation for Trace Internal Commands*/
/**
 * This function is added to send commands from the target 
 * components directly
 *
 * 
 *        buf		 Buffer having the Hex values
 *		  size		 Size of the buffer
 * return \li E_OK   if successful
 *                   else error values returned by concrete implementation.
 */
EXPORT S32 TR_core_s32SendCmd(void* buf, U32 size);

/**
 * This function is to provide the current block mode status
 *
 * \parm  p_u32	TraceBlockMode
 *		p_u32	ProxyBlockMode
 *        
 * return \li 	void
 *                       
 */
IMPORT void TR_get_blockmode_status(U32 *TraceBlockMode,
                                   U32 *ProxyBlockMode);

/**
 * Trace entry point while Trace runs from Resident Unit.
 * Will start the Trace frontend. If necessary, the backend is started as well.
 * The backend is started if it is located in the same loadable unit.
 * In the resident unit or when the backend is located in a different unit, the
 * backend has to be explicitly started using TRACE_backend_main.
 *
 * \parm ac Startup or shutdown info, argument count in startup case
 * \parm av If av[0] is present (ac >= 1) it will be treated as BOOL* and used
 *          to store the information whether the backend was started or not.
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
IMPORT S32 TRACE_main(S32 ac, U8** av);

/**
 * Trace backend entry point while Trace runs from Resident Unit.
 * Will start/stop the Trace backend.
 *
 * \parm ac Startup or shutdown info
 * \parm av Pointer to additional startup info (not used)
 *
 * return \li E_OK    if successful
 *                    else error values returned by concrete implementation.
 */
IMPORT S32 TRACE_backend_main(S32 ac, S8** av);

/**
 * Register with Trace for notification events
 *
 * \parm  msg      ptr to TRACE_notify structure
 *
 * return \li S32   E_OK if successfully started
 *                 else, error as returned by concrete implementation
 */
IMPORT S32 TRACE_reg_notify_evt(TRACE_notify* msg);

/**
 * Unregister with Trace for notification events
 *
 * \parm  msg      ptr to TRACE_notify structure
 *
 * return \li S32   E_OK if successfully started
 *                 else, error as returned by concrete implementation
 */
IMPORT S32 TRACE_unreg_notify_evt(TRACE_notify* msg);

/**
 * Inform Trace about removable media attach evt
 *
 * \parm  msg      ptr to TRACE_media_type structure
 *
 * return \li S32   E_OK if successful
 *                 else, error as returned by concrete implementation
 */
IMPORT S32 TRACE_media_attach(TRACE_media_type mt);

/**
 * Inform Trace about removable media detach evt
 *
 * \parm  msg      ptr to TRACE_media_type structure
 *
 * return \li S32   E_OK if successful
 *                 else, error as returned by concrete implementation
 */
IMPORT S32 TRACE_media_detach(TRACE_media_type mt);

/* This structure is used to check if a pair of trace class and trace level is
 * enabled. This check is useful when a trace message with a large amount of
 * data need to be assembled. The fIsActive is for the result value
 * The same structure is used for tracing out the messsages
 */
typedef struct
{
   U32                  uwLength;   /* Message length                    */
   TR_tenTraceLevel     ubLevel;    /* Trace level for the message       */
   U32                  uhClass;    /* Trace class for the message       */
   S8*                  p_ubBuffer; /* The buffer containing the message */
   BOOL                 fIsActive;  /* Class/level active status         */
}TraceData;

/* This structure is used for Tracing out Large Binary streams
 * by special socket clients
 * This can be used to trace out messages bigger than normal(239 bytes)trace msg
 * Currently supports stream of Max size 3543 bytes (3KB)
 * NOTE: ONLY SPECIAL CLIENTS can receive these msg
 */
typedef struct
{

  U32 uwCompId;   /* Trace Component ID                                      */
                  /* Can be ignored for ADIT platforms added for
                     compatibility with BP                                   */
  U32 uwSockId;   /* Socket Port(Address) of TTFis Client                    */
  U32 uwLength;   /* Length of the Message(32 bit)                           */
  U8* p_ubBuffer; /* Buffer containing the message                           */
}TraceStreamData;

/* This structure is added for the compatibility with BP */
typedef struct
{
   U8   fIsActive;                      /* Result value */
   TR_tenTraceClass uhTraceClass;       /* Trace class  */
   TR_tenTraceLevel ubTraceLevel;       /* Trace level  */
}ActiveTrace;

typedef struct
{
  TR_tenTraceChan  enTraceChannel;   /* ChannelID         */
  TRACE_CALLBACK   p_Callback;       /* Callback function */
} LaunchChannel;

/**
 * "Backward Compatibility" wrapper for trace frontend startup
 */
IMPORT S32 trace_drv_core_init (void);
/**
 * "Backward Compatibility" wrapper for trace shutdown
 */
IMPORT S32 trace_drv_core_uninit(void);

/**
 * "Backward Compatibility" wrapper for trace backend startup
 */
IMPORT S32 trace_drv_init(void);

/**
 * "Backward Compatibility" wrapper for trace backend shutdown
 */
IMPORT S32 trace_drv_uninit(void);

#ifdef __cplusplus
}
#endif
#endif
